// PE1MUD 24/3/2025 V1.2
//
// Remove the controller from your board as applicable (depends on board type).
//
// Connect the Arduino pins as described below.
//
// The RTC6705 updates it's frequency very ugly, therefore, the frequency
// only gets updated after a 1.5 second wait.
//
// On many modules, the power control seems not to work.
// So... there is no power control. Yet.
//
// Have fun!
//

// Ensure, in your RTC6705, Pin1 is tied to 3V3 to enable the SPI interface.
#define CLK     2 // Pin 7 on RTC6705
#define LE      3 // Pin 6 on RTC6705
#define DATA    4 // Pin 5 on RTC6705

// Rotary connections: 100n on pin, pullup with 10k to 5V, connect to rotary by 220R.
#define PIN_IN1 5 // Connect your rotary pin1 here via the 220R
#define PIN_IN2 6 // Connect your rotary pin2 here via the 220R
#define PIN_SW  7 // Connect your push switch here (on the rotary) via the 220R
// Solder center pin from rotary to ground, other side of switch too

#include <LiquidCrystal_I2C.h>    // Library 1.1.2 by Frank de Brabander
#include <RotaryEncoder.h>        // Library 1.5.3 by Matthias Hertel
#include <EEPROM.h>

#define LOPO  0x04c7d // Low power config
#define MEPO  0x04d0d // Medium power config
#define HIPO  0x04fbd // Default high power config
#define TEPO  0x04fbd // Test value

RotaryEncoder *encoder = nullptr;

// at 50ms or slower, there should be no acceleration
uint8_t cutoff = 50;
uint8_t rotsw = 1;
uint8_t mem = 0;

uint32_t freq[2];

uint16_t timeout = 1; // at 0, update lcd and rtc6075
uint8_t elapsed = 0;

LiquidCrystal_I2C lcd(0x27,20,4);  // set the LCD address to 0x27 for a 16 chars and 2 line display, connect to A4 (SDA) and A5 (SCL)
// or 0x3F, depending on your lcd....

void checkPosition()
{
  encoder->tick(); // just call tick() to check the state.
}

void showFreq(void)
{
  lcd.clear();
  lcd.print("RTC6705 Ctl V1.2");
  lcd.setCursor(0, 1);
  lcd.print("Freq");
  lcd.print(mem+1);
  lcd.print(": ");
  lcd.print(freq[mem]/1000);
  lcd.print(" MHz");
  if (timeout > 1) lcd.print("*");
}

void setTX(void)
{
  uint32_t n_reg;
  uint32_t a_reg;
  n_reg = (freq[mem] / 2560); // divided by 64 * 20.000kHz ref * 2 = 2560
  a_reg = (freq[mem] - (n_reg * 2560)) / 40; // calculate the remainder
  
  send(0x1f, 0L);
  // send(0x17, LOPO); // option: set power level - does not work on the blue TS832
  send(0x11, (n_reg<<7) + a_reg);
  showFreq();
  EEPROM.put(mem*4, freq[mem]); // Update to eeprom
}

void sendbit(uint8_t bit)
{
  digitalWrite(DATA, bit);
  delayMicroseconds(5);
  digitalWrite(CLK, HIGH);
  delayMicroseconds(5);
  digitalWrite(CLK, LOW);
}

void send(uint8_t address, uint32_t registers)
{
  uint8_t count;

  digitalWrite(LE, LOW);
  delayMicroseconds(5);

  // Send address
  for (count = 0; count < 5; count++)
  {
    if ((address & 0x01)==0x01)
      sendbit(HIGH);
    else
      sendbit(LOW);
    address = address >> 1;
  }
  // Send register bits
  for (count = 0; count < 20; count++)
  {
    if ((registers & 0x01)==0x01)
     sendbit(HIGH);
    else
      sendbit(LOW);
    registers = registers >> 1;
  }
  delayMicroseconds(5);
  digitalWrite(LE, HIGH);
}

void setup()
{
  Serial.begin(115200); // open the serial port at 115200 bps:
  lcd.init();                      // initialize the lcd 
  lcd.clear();
  lcd.backlight();

  for (mem=0; mem<2; mem++)
  {
    EEPROM.get(mem*4, freq[mem]);
    if (freq[mem] > 5850000 || freq[mem] < 5650000)
    {
      freq[mem] = 5725000;
      EEPROM.put(mem*4, freq[mem]);
    }
  }

  EEPROM.get(8, mem);
  if (mem>1) mem = 0;

  // setup the rotary encoder functionality
  encoder = new RotaryEncoder(PIN_IN1, PIN_IN2, RotaryEncoder::LatchMode::FOUR3);

  // register interrupt routine
  attachInterrupt(digitalPinToInterrupt(PIN_IN1), checkPosition, CHANGE);
  attachInterrupt(digitalPinToInterrupt(PIN_IN2), checkPosition, CHANGE);

  digitalWrite(CLK, LOW);
  digitalWrite(DATA, LOW);
  digitalWrite(LE, HIGH);
  
  pinMode(CLK, OUTPUT);
  pinMode(DATA, OUTPUT);
  pinMode(LE, OUTPUT);
  pinMode(PIN_SW, INPUT);
  rotsw = digitalRead(PIN_SW);
}

void loop()
{
  int direction;

  delay(1);
  if (elapsed < 100) elapsed++;

  if (timeout)
  {
    if(timeout==1) setTX();
    timeout--;
  }

  if (rotsw != digitalRead(PIN_SW))
  {
    if (rotsw == 1) 
    {
      if (timeout > 0) // If timeout is running while memory is switched, commit to eeprom
      {
        EEPROM.put(mem*4, freq[mem]);
        timeout =0;
      }
      mem = (mem == 0? 1:0);
      EEPROM.put(8, mem);
      setTX();
    }

    rotsw = digitalRead(PIN_SW);
  }

  encoder->tick(); // just call tick() to check the state.

  direction = (int)encoder->getDirection();
  if (direction!=0)
  {
    timeout = 1500;
    // accelerate when there was a previous rotation in the same direction.
    if (elapsed < cutoff) direction = 5 * direction;
    freq[mem] = freq[mem] + (uint32_t)direction * 1000;
    if (freq[mem] > 5850000) freq[mem] = 5650000;
    if (freq[mem] < 5650000) freq[mem] = 5850000;
    showFreq();
    elapsed = 0;
  }
}

